%% PLUS Joint (Matrix)
% Script to combine results from individual benchmarks in a joint form.
%
% Torben Larsen, Aalborg University, Denmark. tl@es.aau.dk. 13-FEB-2010.
% http://www.accelereyes.com/wiki/index.php?title=Torben's_Corner
%
%     0.1:  13-FEB-2010   -Basic functionality.
%
% The .m file MUST be named in the following way:
%
%    FCT_Joint_ATP_PRC.m  >>  FCT_Joint_M_PRC.m
%
% where:
%
%   FCT:     Function to be benchmarked
%   ATP:     Array type (MUST be M here - Matrix)
%   PRC:     Precision (SP or DP - Single Precision or Double Precision)
%
% Currently the following CPU and GPUs are used:
%   Ref.:    The reference is the Core 2 Duo 2.8 GHz (C2D28)
%   GPU1:    NVIDIA C1060 Tesla
%   GPU2:    NVIDIA 9800GT
%   GPU3:    NVIDIA 9600M GT
%
clear all;


%% FILE NAME EXTRACTION
S = regexp(mfilename, '_', 'split');
FCT   = char(S(1));     % Function to be benchmarked
ATP   = char(S(3));     % Array type (Matrix/Vector)
PRC   = char(S(4));     % Precision (Single Precision / Double Precision)
if ~strcmp(ATP,'M'),
    error('This benchmark file only applies to Matrix analysis');
end


%% LOAD DATA FOR REFERENCE CPU
Name = [FCT '_9400M_C2D28_OSXbt_M_SP.mat'];
if exist(Name,'file')~=2, error(['Missing file: ' Name]); end
eval(['load ' Name]);
Size_9400M  = Size;
T_CPU_C2D28 = T_CPU;


%% LOAD DATA FOR 9600M GT GPU
Name = [FCT '_9600MGT_C2D28_OSXhp_M_SP.mat'];
if exist(Name,'file')~=2, error(['Missing file: ' Name]); end
eval(['load ' Name]);
Size_9600MGT  = Size;
T_GPU_9600MGT = T_GPU;


%% LOAD DATA FOR 9800GT GPU
Name = [FCT '_9800GT_i7x975_W7_M_SP.mat'];
if exist(Name,'file')~=2, error(['Missing file: ' Name]); end
eval(['load ' Name]);
Size_9800GT  = Size;
T_CPU_i7x975 = T_CPU;
T_GPU_9800GT = T_GPU;


%% LOAD DATA FOR TESLA GPU
Name = [FCT '_Tesla_i7x975_W7_M_SP.mat'];
eval(['load ' Name]);
Size_Tesla  = Size;
T_GPU_Tesla = T_GPU;


%% PLOT RESULTS AND SAVE FIGURE
% Create figure and first plot
figure(1); clf(1);
plot(Size_9400M, T_CPU_C2D28./T_CPU_i7x975, 'c:*', ...
     Size_9400M, T_CPU_C2D28./T_GPU_9600MGT, 'r-*', ...
     Size_9400M, T_CPU_C2D28./T_GPU_9800GT, 'b-*', ...
     Size_9400M, T_CPU_C2D28./T_GPU_Tesla, 'g-*', ...
     'LineWidth', 2);
grid;
    
% Make title
PRCL = 'Single Precision (SP)';
if strcmp(PRC,'DP'), PRCL = 'Double Precision (DP)'; end
F_TTL = [FCT ' - ' PRCL ', Matrix - Ref.: Core 2 Duo 2.8GHz'];
title(F_TTL);
    
% Labels & Legends
xlabel('#Rows = #Columns   [-]');
ylabel('Speed-up   [-]');
legend('Core i7-975', '9600M GT', '9800GT', 'Tesla', ...
    'Location', 'NorthWest');

%  Save figures
jpgFname = [FCT '_Joint_M_SP.jpg'];
print( gcf, '-djpeg99', '-r100', jpgFname );