<?php
/** This file not safe to copy from sandbox to live directly, ask Joe */

/**
 * Collection of helpful functions to hold common static logic for the purchase
 * process.  These are all pretty specific to our own AccelerEyes modifications,
 * particularly the addition of Licenses.
 */
class CustomPurchaseHelpers {

  // THESE PIDS DIFFER BETWEEN LIVE AND SANDBOX
  
  // base license and features
  const PID_BASE = 105;
  const PID_DLA = 101;
  const PID_SDK = 106;
  const PID_JMC = 107;
  const PID_MGL = 108;
  const PID_SLA = 146;
  
  // items used when upgrading sms/phone of existing licenses
  const PID_SMS_UPGRADE = 139;
  const PID_PHONE_UPGRADE = 140;  

  // Libjacket pids
  const LJ_PID_FIVE   = 000;
  const LJ_PID_TEN    = 000;
  const LJ_PID_BUNDLE = 000;
  const LJ_PID_BASE   = 152;
  const LJ_PID_DLA    = 000;
  const LJ_PID_MGL    = 000;
  const LJ_PID_SLA    = 000;
  const LJ_PID_HPC    = 000;

  // Libjacket maintenance pids
  const LJ_PID_FIVE_MAINT   = 000;
  const LJ_PID_TEN_MAINT    = 000;
  const LJ_PID_BUNDLE_MAINT = 000;
  const LJ_PID_BASE_MAINT   = 000;
  const LJ_PID_DLA_MAINT    = 000;
  const LJ_PID_MGL_MAINT    = 000;
  const LJ_PID_SLA_MAINT    = 000;
  const LJ_PID_HPC_MAINT    = 000;

  // Libjacket phone pids
  const LJ_PID_FIVE_PHONE   = 000;
  const LJ_PID_TEN_PHONE    = 000;
  const LJ_PID_BUNDLE_PHONE = 000;
  const LJ_PID_BASE_PHONE   = 000;
  const LJ_PID_DLA_PHONE    = 000;
  const LJ_PID_MGL_PHONE    = 000;
  const LJ_PID_SLA_PHONE    = 000;
  const LJ_PID_HPC_PHONE    = 000;

  // Libjacket CNL pids
  const LJ_PID_FIVE_CNL   = 000;
  const LJ_PID_TEN_CNL    = 000;
  const LJ_PID_BUNDLE_CNL = 000;
  const LJ_PID_BASE_CNL   = 000;
  const LJ_PID_DLA_CNL    = 000;
  const LJ_PID_MGL_CNL    = 000;
  const LJ_PID_SLA_CNL    = 000;
  const LJ_PID_HPC_CNL    = 000;
  
  /**
   * When passed a valid License instance, will return the price premium that new
   * features purchased for this existing license should have applied.  This is
   * influenced by the maintenance and support dates on the license.
   */
  public static function calculatePricePremium($currentLicense){  
    $priceModification = 1;
    
    // a price premium on new features if a support period is active for license
    if ($currentLicense->getSupportDate() != null 
    && ($currentLicense->getSupportDate() > time())){
      $priceModification += ".2";    
    }

    // a price premium on new features if a premium SMS period is active for license
    $yearTime = (365 * 24 * 60 * 60); // not 100% correct in leap year but close enough
    if ($currentLicense->getMaintenanceDate() != null
    && ($currentLicense->getMaintenanceDate() > time())){
      $numYearsLeft = ceil(($currentLicense->getMaintenanceDate() - ($currentLicense->getPurchaseDate() + $yearTime)) / $yearTime);
      if ($numYearsLeft > 0){
        $priceModification += (.1 * $numYearsLeft);
      }
    }
    
    return $priceModification;
  }
  
  /**
   * Returns the number of years between the previous maintenance date on a license
   * and the current day.  This is the number of years a user must pay for to add 
   * SMS to an existing license where maintenance has lapsed
   */
  public static function maintenanceYearsRequired($license){
    $yearTime = (365 * 24 * 60 * 60);
    
    if ($license == null) return 1;
    if ($license->getMaintenanceDate() == null){
      $license->setMaintenanceDate($license->getPurchaseDate());
    }
    if ($license->getMaintenanceDate() > time()){
      // maintenance expires in future, add a year to the end
      return 1;
    }
    
    $diff = floor((time() - $license->getMaintenanceDate()) / $yearTime);
    
    return $diff + 1;
  }

  /**
   * Whether or not the current user is a student.
   */
  public static function isStudentUser(){
    $groupId = Mage::getSingleton('customer/session')->getCustomerGroupId();
    $collection = Mage::getModel('customer/group')->getCollection();
  
    foreach ($collection as $group){
      $_g = $group->toArray(array('customer_group_id', 'customer_group_code'));
      $groups[$_g['customer_group_id']] = $_g['customer_group_code'];
    }
  
    $groupName = $groups[$groupId];
    
    return $groupName == 'Student';
  }
  
  /**
   * Generates a feature list for a license
   */
  public static function featureString($lic){
    $features = "";
    if ($lic['mgl'] > 0) $features .= "mgl" . $lic['mgl'];
    if ($lic['jla'] > 0){
      if (strlen($features) > 0) $features .= ", ";
      $features .= "dla";
    }
    if ($lic['sdk'] > 0){
      if (strlen($features) > 0) $features .= ", ";
      $features .= "sdk";
    }
    if ($lic['jmc'] > 0){
      if (strlen($features) > 0) $features .= ", ";
      $features .= "jmc";
    }  
    
    return strtoupper($features);
  }
  
  /** 
   * Generates a label for displaying a license in a list
   */
  public static function licenseSelectLabel($license){
    $label = $license['license_number'];
    
    if (!empty($license['host_label'])){
      $label .= " (" . $license['host_label'] . ")";
    }
    
    $label .= " purchased on " . date("m/d/Y", $license['purchase_date']);
    
    return $label;
  }
  
  /**
   * Generates html for a date, marked red if null or in the past
   */
  public static function expirationDate($timestamp){
    if (empty($timestamp) || time() > $timestamp){
      return "<span style=\"color:red\">EXPIRED</span>";
    }  
    
    return "<span>" . date("m/d/Y", $timestamp) . "</span>";
  }
  
  /**
   * Returns the original price of a license.  The various ids of
   * of features are necessary because the production and sandbox 
   * databases are very different.
   *
   * NOTE: This ONLY includes features, not SMS or PHONE
   */
  public static function originalPrice($license){
    $catalogModel = Mage::getModel('catalog/product');
  
    $price = $catalogModel->load(self::PID_BASE)->getTierPrice('price');
    
    if ($license['jla'] > 0) 
      $price += $catalogModel->load(self::PID_DLA)->getTierPrice('price');
    if ($license['sdk'] > 0) 
      $price += $catalogModel->load(self::PID_SDK)->getTierPrice('price');
    if ($license['jmc'] > 0) 
      $price += $catalogModel->load(self::PID_JMC)->getTierPrice('price');
    
    if ($license['mgl'] > 0){
      $price += ($license['mgl'] - 1) * $catalogModel->load(self::PID_MGL)->getTierPrice('price');
    }
    
    return $price;
  }
  
  /**
   * Returns the tier price for the PID specified.  It is up to the caller
   * to pass a valid PID.
   */
  public static function tierPrice($pid){
    if ($pid == null) return null;
    
    $catalogModel = Mage::getModel('catalog/product');
    
    return $catalogModel->load($pid)->getTierPrice('price');
  }
  
  /**
   * Generates the HTML for a product row
   */
  public static function productRow($code, $name, $id, $description = null, $toolTip = null, $onChange = null, $priceMod = 1){
    $catalogModel = Mage::getModel('catalog/product');
  
    if ($description == null) $description = $catalogModel->load($id)->getDescription();
    if ($toolTip == null) $toolTip = $catalogModel->load($id)->getDescription();
    
    if ($onChange == null) $onChange = "updateTotal()";
  
    $html  = "<tr>";
    $html .= "<td class=\"purchase-product-title\">" . $name . "</td>";
    $html .= "<td class=\"purchase-product-desc\">";
    $html .=   "<span id=\"" . $code . "-desc\">" . $description . "</span>";
    $html .=   "<div id=\"" . $code . "-desc-tooltip\" class=\"purchase-tooltip\" style=\"display:none\">";
    $html .=     $toolTip . "<br /></div></td>";
  
    $html .= "<td class=\"purchase-product-qty\">";
    
    if ($code == "mgl"){
      $html .= "<input class=\"input-text\" id=\"" . $code . "-qty\" name=\"" . $code
        . "-qty\" value=\"0\" onchange=\"" . $onChange . "\"/>";
    }
    else{
      $html .= "<div style=\"position:relative; left: 60px; margin: auto\">";
      $html .= "<input type=\"checkbox\" class=\"input-checkbox\" id=\"" . $code . "\" name=\"" . $code . "\" onchange=\"" . $onChange . "\">";
      $html .= "</div>";
    }
    
    $html .= "</td>";
    
    $html .= "<td class=\"purchase-product-price\">";
    $html .=   "<p class=\"purchase-product-price-par\">";
    $html .= Mage::helper('core')->currency($catalogModel->load($id)->getTierPrice('price') * $priceMod);
    $html .=   "</p>";
    
    $html .=   "<input type=\"hidden\" name=\"" . $code . "-price\" id=\"" . $code . "-price\" "
                        . "value=\"" . $catalogModel->load($id)->getTierPrice('price') . "\">";
  
    $html .= "</td></tr>";
    
    return $html;
  }
  
}

?>